//*****************************************************************************
// File Name	: diglevel.c
// 
// Title		: digital level
// Revision		: 1.0
// Notes		:	
// Target MCU	: Atmel AVR series
// Editor Tabs	: 4
// 
// Revision History:
// When			Who			Description of change
// -----------	-----------	-----------------------
// 13-Jul-2004	pstang		Created the program
//*****************************************************************************


//----- Include Files ---------------------------------------------------------
#include <avr/io.h>		// include I/O definitions (port names, pin names, etc)
#include <avr/interrupt.h>	// include interrupt support
#include <avr/eeprom.h>	// eeprom support
#include <math.h>

#include "global.h"		// include our global settings
#include "uart.h"		// include uart function library
#include "rprintf.h"	// include printf function library
#include "timer.h"		// include timer function library (timing, PWM, etc)
#include "a2d.h"		// include A/D function library
#include "lcd.c"		// include LCD support
#include "i2c.h"		// include I2C function library
#include "lis3l02.h"	// include LIS3L02 accelerometer support

// defines

// global variables
s32 Xa;
s32 Ya;
s32 Za;

s32 Xbias=0;
s32 Ybias=0;
s32 Zbias=0;
s32 Xfs1g=0;
s32 Yfs1g=0;
s32 Zfs1g=0;

// functions
void digprint(void);
void diglevel(void);
void sample(void);
void calibrate(void);

//----- Begin Code ------------------------------------------------------------
int main(void)
{
	// initialize our libraries
	// initialize the UART (serial port)
	uartInit();
	uartSetBaudRate(115200);
	// make all rprintf statements use uart for output
	rprintfInit(uartSendByte);
	// initialize the timer system
	timerInit();
	// initialize i2c
	i2cInit();
	i2cSetBitrate(100);
	// set pullup resistors
	sbi(PORTC,0);
	sbi(PORTC,1);

	// print welcome message
	rprintfProgStrM("\r\nDigital Accelerometer Test Suite\r\n");

	// sensor init and check
	rprintf("Initializing sensor...");
	if(!lis3l02Init())
	{
		rprintf("OK\n");
	}
	else
	{
		rprintf("Init Failed!\n");
	}
	timerPause(500);
	
	// run example
	//digprint();		// print digital readings
	
	diglevel();	// digital level example

	return 0;
}

void digprint(void)
{
	// intro
	rprintfProgStrM("Printing sensor readings...\n");
	timerPause(500);

	// run
	while(1)
	{
		// sample the sensor
		sample();

		// print the raw results to the serial port
		rprintf("  X:"); rprintfNum(10,6,TRUE,' ',Xa);
		rprintf("  Y:"); rprintfNum(10,6,TRUE,' ',Ya);
		rprintf("  Z:"); rprintfNum(10,6,TRUE,' ',Za);
		rprintfCRLF();

		// wait 50ms before beginning next sample cycle
		timerPause(50);
	}

}

void diglevel(void)
{
	s16 Xtilt;
	s16 Ytilt;
	double Xc;
	double Yc;
	double Zc;

	// intro
	rprintfProgStrM("Digital Level Example\n");
	timerPause(500);

	// self-calibrate
	rprintf("Calibrating for level reading...\n");
	calibrate();
	rprintf("Done!\n");
	timerPause(500);

	// run
	while(1)
	{
		// sample the sensor
		sample();

		// print the raw results to the serial port
		rprintf("  X:"); rprintfNum(10,6,TRUE,' ',Xa);
		rprintf("  Y:"); rprintfNum(10,6,TRUE,' ',Ya);
		rprintf("  Z:"); rprintfNum(10,6,TRUE,' ',Za);

		// gravity-vector angle calculation
		
		// apply scale factor
		Xc = (double)Xa/Xfs1g;
		Yc = (double)Ya/Yfs1g;
		Zc = (double)Za/Zfs1g;

		// limit input range for trig functions to follow
		Xc = MIN(Xc, 1);
		Xc = MAX(Xc, -1);
		Yc = MIN(Yc, 1);
		Yc = MAX(Yc, -1);

		// calculate vector angles
//		Xtilt = asin(Xc)*((180*10)/PI);			// asin method
//		Ytilt = asin(Yc)*((180*10)/PI);			// asin method
		Xtilt = atan2(Xc,Zc)*(180/M_PI);	// atan method
		Ytilt = atan2(Yc,Zc)*(180/M_PI);	// atan method

		// print results to LCD display
		rprintf(" | ");
		rprintf("  Xtilt= "); rprintfNum(10,4,TRUE,' ',Xtilt);
		rprintf("  Ytilt= "); rprintfNum(10,4,TRUE,' ',Ytilt);
		rprintf(" degrees");
		rprintfCRLF();

		// print results to LCD display with bargraph
//		lcdGotoXY(0,0); rprintf("X"); rprintfNum(10,4,TRUE,' ',Xtilt); rprintfChar('<'); lcdProgressBar((Xtilt+100),200,9); rprintfChar('>');
//		lcdGotoXY(0,1); rprintf("Y"); rprintfNum(10,4,TRUE,' ',Ytilt); rprintfChar('<'); lcdProgressBar((Ytilt+100),200,9); rprintfChar('>');
//		lcdGotoXY(0,0); rprintf("X"); rprintfNum(10,5,TRUE,' ',Xc*1000);
//		lcdGotoXY(0,1); rprintf("Y"); rprintfNum(10,5,TRUE,' ',Yc*1000);
		
		// wait 50ms before beginning next sample cycle
		timerPause(50);
	}

}


void sample(void)
{
	u08 i;
	Xa = 0;
	Ya = 0;
	Za = 0;

	// average over 50 readings
	for(i=0; i<50; i++)
	{
		Xa += lis3l02GetAccel(0)+Xbias;
		Ya += lis3l02GetAccel(1)+Ybias;
		Za += lis3l02GetAccel(2)+Zbias;
	}

	// normalize the average
	Xa /= 50;
	Ya /= 50;
	Za /= 50;
}

void calibrate(void)
{
	// this calibration routine assumes that the sensor is level and sensing 1g in the Z axis
	// additional orientations are needed for complete calibration

	// sample the sensor
	sample();
	
	// record bias in X,Y
	Xbias = -Xa;
	Ybias = -Ya;
	// cannot resolve bias in Z using this sensor orientation, so record zero bias
	Zbias = 0;

	// set scale factors for X,Y,Z
//	Xfs1g = lis3l02GetAccel(2);
//	Yfs1g = lis3l02GetAccel(2);
//	Zfs1g = lis3l02GetAccel(2);
//	Xfs1g = 16384;
//	Yfs1g = 16384;
	Xfs1g = 16380;
	Yfs1g = 16380;
	Zfs1g = 16380;
}
